/* Copyright (c) <2012>, Intel Corporation
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 *  - Neither the name of Intel Corporation nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software 
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include "opengl.h"
#include "system.h"
#include "CPUTMath.h"
#include "shader.h"
#include "model.h"
#include "rendertarget.h"
#include "App.h"

static float resolution_factor = 0.75f;
static float rotation_factor = 0.1f;

static model_t* train_station = NULL;
static const float scale_factor = 0.001f;

unsigned int screen_width;
unsigned int screen_height;
float aspect_ratio;

float4x4 world_matrix;
float4x4 view_matrix;
float4x4 projection_matrix;
float4x4 view_projection_matrix;

// fullscreen quad
model_t* fs_quad_model;
GLuint fs_quad_shader_program;
GLuint fs_quad_texture_location;
GLint fs_quad_position_attrib;
GLint fs_quad_texture_attrib;

render_target_t* render_target;

//--------------------------------------------------
// Initalize the app
// It's assumed that OpenGL is initialized and ready to do
//--------------------------------------------------
void app_init( unsigned int width, unsigned int height )
{
    screen_width = width;
    screen_height = height;
    aspect_ratio = (float)screen_width / (float)screen_height;
    
    glClearColor(0.8f, 0.8f, 0.8f, 1.0f);
    glClearDepthf(1.0f);
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LEQUAL);
    glEnable(GL_CULL_FACE);
    glCullFace(GL_BACK);
    
    // load a model
    train_station = load_mesh( "assets/models/train_station.mesh" );
    
    // init matrices
    float4x4 scale_matrix = float4x4Scale(scale_factor, scale_factor, scale_factor);
    float4x4 rotation_matrix = float4x4RotationY(kPiDiv2);
    world_matrix = float4x4multiply(&scale_matrix, &rotation_matrix);
    view_matrix = float4x4translation(0.0f, 0.0f, 15.0f);
    view_projection_matrix = float4x4identity();
    
    projection_matrix = float4x4PerspectiveFovLH(DegToRad(65.0f), aspect_ratio, 0.1f, 100.0f);
    
    // Create offscreen render target
    render_target = create_render_target( screen_width, screen_height );
    
    // fullscreen quad
    fs_quad_model = create_quad_model( 0 );
    fs_quad_shader_program = load_shader("assets/shaders/fullscreen_quad.shader");
    fs_quad_position_attrib = glGetAttribLocation(fs_quad_shader_program, "a_position");
    fs_quad_texture_attrib = glGetAttribLocation(fs_quad_shader_program, "a_texcoord");
    fs_quad_texture_location = glGetUniformLocation(fs_quad_shader_program, "s_diffuse");
}

//--------------------------------------------------
// Update the frame
//--------------------------------------------------
void app_update( float fElapsedTime )
{
    static float total_time = 0;
    
    float4x4 rotation_y = float4x4RotationY( rotation_factor * fElapsedTime );
    world_matrix = float4x4multiply( &world_matrix, &rotation_y );
    
    total_time += fElapsedTime;
}

//--------------------------------------------------
// Render the frame
//--------------------------------------------------
void app_render( void )
{
    // 1. SAVE OUT THE DEFAULT FRAME BUFFER
    static GLint default_frame_buffer = 0;
    glGetIntegerv(GL_FRAMEBUFFER_BINDING, &default_frame_buffer);
    
    // 2. RENDER TO OFFSCREEN RENDER TARGET
    glBindFramebuffer(GL_FRAMEBUFFER, render_target->frame_buffer);
    glViewport(0, 0, render_target->width * resolution_factor, render_target->height * resolution_factor);
    glClearColor(0.0f, 0.25f, 0.25f, 1.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
    view_projection_matrix = float4x4multiply( &view_matrix, &projection_matrix );
    draw_model( train_station, &view_projection_matrix, &world_matrix );
    
    // 3. RESTORE DEFAULT FRAME BUFFER
    glBindFramebuffer(GL_FRAMEBUFFER, default_frame_buffer);
    glBindTexture(GL_TEXTURE_2D, 0);
    
    // 4. RENDER FULLSCREEN QUAD
    glViewport(0, 0, screen_width, screen_height);
    glClearColor(1.0f, 0.0f, 0.0f, 1.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    
    glUseProgram(fs_quad_shader_program);
	glEnableVertexAttribArray( fs_quad_position_attrib );
	glEnableVertexAttribArray( fs_quad_texture_attrib );
    
    glBindBuffer(GL_ARRAY_BUFFER, fs_quad_model->positions_buffer);
    glVertexAttribPointer(fs_quad_position_attrib, 3, GL_FLOAT, GL_FALSE, 0, 0);
    
    glBindBuffer(GL_ARRAY_BUFFER, fs_quad_model->texcoords_buffer);
    
    const float2 texcoords_array[] =
    {
        { resolution_factor, resolution_factor },
        { 0.0f,              resolution_factor },
        { 0.0f,              0.0f              },
        { resolution_factor, 0.0f              },
    };
    glBufferData(GL_ARRAY_BUFFER, sizeof(float3) * fs_quad_model->num_vertices, texcoords_array, GL_STATIC_DRAW);
    
    glVertexAttribPointer(fs_quad_texture_attrib, 2, GL_FLOAT, GL_FALSE, 0, 0);

	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, fs_quad_model->index_buffer );

    glActiveTexture(GL_TEXTURE0);
    glBindTexture(GL_TEXTURE_2D, render_target->texture);
    glUniform1i(fs_quad_texture_location, 0);
    
    glDrawElements(GL_TRIANGLES, fs_quad_model->num_indices, GL_UNSIGNED_INT, 0);
    
    // Unbind buffer before we return
    glBindBuffer( GL_ARRAY_BUFFER, 0 );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, 0 );
    glBindTexture(GL_TEXTURE_2D, 0);
}

//--------------------------------------------------
// Utility functions
//--------------------------------------------------

void app_set_resolution( float fResolution )
{
    resolution_factor = fResolution;
}

void app_set_rotation( float fRotation )
{
    rotation_factor = fRotation;
}

void app_set_filtering( int iFiltering )
{
    set_filtering( render_target, iFiltering );
}
