/** @file
  The CPU specific programming for PiSmmCpuDxeSmm module, such as SMRR.
  Currently below CPUs are supported.

  0x00030670 // VALLEYVIEW

  Copyright (c) 2013-2015 Intel Corporation.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  * Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in
  the documentation and/or other materials provided with the
  distribution.
  * Neither the name of Intel Corporation nor the names of its
  contributors may be used to endorse or promote products derived
  from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

**/

#include <Base.h>

#include <Library/PcdLib.h>
#include <Library/BaseLib.h>
#include <Library/CpuLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/DebugLib.h>
#include <Library/PciLib.h>
#include <Library/LocalApicLib.h>

#include "PiSmmCpuDxeSmm.h"
#include "SmmFeatures.h"
#include "SmmStm.h"

//
// The CPUID mapping for Silvermont
//
CPUID_MAPPING mSilvermontMap[] = {
  {CPUID_SIGNATURE_VALLEYVIEW, CPUID_MASK_NO_STEPPING},         // ValleyView
};

//
// The CLASS for Silvermont
//
CPU_SMM_CLASS mSilvermontClass = {
  CpuSilvermont,
  sizeof(mSilvermontMap)/sizeof(mSilvermontMap[0]),
  mSilvermontMap,
  };

//
// This table defines supported CPU class
//
CPU_SMM_CLASS *mCpuClasstable[] = {
  &mSilvermontClass,
  };

////////
// Below section is common definition
////////

CPU_SMM_FEATURE_CONTEXT  mFeatureContext;
CPU_SMM_CLASS            *mThisCpu;

/**
  Return if SMRR is supported

  @retval TRUE  SMRR is supported.
  @retval FALSE SMRR is not supported.

**/
BOOLEAN
IsSmrrSupported (
  VOID
  )
{
  UINT64                            MtrrCap;

  MtrrCap = AsmReadMsr64(EFI_MSR_IA32_MTRR_CAP);
  if ((MtrrCap & IA32_MTRR_SMRR_SUPPORT_BIT) == 0) {
    return FALSE;
  } else {
    return TRUE;
  }
}

////////
// Below section is definition for CpuNehalem
////////

/**
  Initialize MSEG in SMM relocate.

  @param  MsegBase           The base address MSEG.
  @param  IsBsp              If this processor treated as BSP.
**/
VOID
NehalemInitMseg (
  IN UINT32  MsegBase,
  IN BOOLEAN IsBsp
  )
{
  STM_HEADER                *StmHeader;

  if ((PcdGet32 (PcdCpuMsegSize) == 0) || (MsegBase == 0)) {
    //
    // MSEG disabled
    //
    return ;
  }

  StmHeader = (STM_HEADER *)(UINTN)MsegBase;

  //
  // If STM image is loaded, we need set valid bit for S3 resume consideration.
  //
  if (StmHeader->HwStmHdr.StmHeaderRevision == (UINT32) RShiftU64 (AsmReadMsr64 (IA32_VMX_MISC_MSR_INDEX), 32)) {
    AsmWriteMsr64 (IA32_SMM_MONITOR_CTL_MSR_INDEX, MsegBase | IA32_SMM_MONITOR_VALID);
  } else {
//    AsmWriteMsr64 (IA32_SMM_MONITOR_CTL_MSR_INDEX, MsegBase);
    // BUGBUG: SNB can not write 0x9B twice
    AsmWriteMsr64 (IA32_SMM_MONITOR_CTL_MSR_INDEX, MsegBase | IA32_SMM_MONITOR_VALID);
  }
}

/**
  Initialize SMRR in SMM relocate.

  @param  SmrrBase           The base address SMRR.
  @param  SmrrSize           The size of SMRR.
**/
VOID
NehalemInitSmrr (
  IN UINT32                SmrrBase,
  IN UINT32                SmrrSize
  )
{
  AsmWriteMsr64 (EFI_MSR_NEHALEM_SMRR_PHYS_BASE, SmrrBase | CACHE_WRITE_BACK);
  AsmWriteMsr64 (EFI_MSR_NEHALEM_SMRR_PHYS_MASK, (~(SmrrSize - 1) & EFI_MSR_SMRR_MASK)); // Valid bit will be set in ConfigSmrr() at first SMI
}

/**
  Configure SMRR register at each SMM entry.
**/
VOID
NehalemConfigSmrr (
  VOID
  )
{
  UINT64 SmrrMask;

  SmrrMask = AsmReadMsr64 (EFI_MSR_NEHALEM_SMRR_PHYS_MASK);
  if ((SmrrMask & EFI_MSR_SMRR_PHYS_MASK_VALID) == 0) {
    AsmWriteMsr64(EFI_MSR_NEHALEM_SMRR_PHYS_MASK, SmrrMask | EFI_MSR_SMRR_PHYS_MASK_VALID);
  }
}

////////
// Below section is definition for the supported class
////////

/**
  This function will return current CPU_SMM_CLASS accroding to CPUID mapping.

  @return The point to current CPU_SMM_CLASS

**/
CPU_SMM_CLASS *
GetCpuFamily (
  VOID
  )
{
  UINT32         ClassIndex;
  UINT32         Index;
  UINT32         Count;
  CPUID_MAPPING  *CpuMapping;
  UINT32         RegEax;

  AsmCpuid (EFI_CPUID_VERSION_INFO, &RegEax, NULL, NULL, NULL);
  for (ClassIndex = 0; ClassIndex < sizeof(mCpuClasstable)/sizeof(mCpuClasstable[0]); ClassIndex++) {
    CpuMapping = mCpuClasstable[ClassIndex]->MappingTable;
    Count = mCpuClasstable[ClassIndex]->MappingCount;
    for (Index = 0; Index < Count; Index++) {
      if ((CpuMapping[Index].Signature & CpuMapping[Index].Mask) == (RegEax & CpuMapping[Index].Mask)) {
        return mCpuClasstable[ClassIndex];
      }
    }
  }

  // Not found!!! Should not happen
  ASSERT (FALSE);
  return NULL;
}

////////
// Below section is external function
////////

/**
  Disable SMRR register when SmmInit set SMM MTRRs.
**/
VOID
DisableSmrr (
  VOID
  )
{
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
  default:
    return ;
  }
}

/**
  Enable SMRR register when SmmInit restore non SMM MTRRs.
**/
VOID
ReenableSmrr (
  VOID
  )
{
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
  default:
    return ;
  }
}

/**
  Return if it is needed to configure MTRR to set TSEG cacheability.

  @retval  TRUE  - we need configure MTRR
  @retval  FALSE - we do not need configure MTRR
**/
BOOLEAN
NeedConfigureMtrrs (
  VOID
  )
{
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
    return FALSE;
  default:
    return TRUE;
  }
}

/**
  Processor specific hook point at each SMM entry.

  @param  CpuIndex    The index of the cpu which need to check.

**/
VOID
SmmRendezvousEntry (
  IN UINTN  CpuIndex
  )
{
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
    //
    // Configure SMRR register at each SMM entry.
    //
    if (mFeatureContext.SmrrEnabled) {
      NehalemConfigSmrr ();
    }
    //
    // T-state fix
    // NHM & SNB BWG: T-state throttling should be disabled while executing the SMM handler.
    //
    gSmmCpuPrivate->TstateMsr[CpuIndex] = AsmReadMsr64 (EFI_MSR_IA32_CLOCK_MODULATION);
    AsmWriteMsr64 (EFI_MSR_IA32_CLOCK_MODULATION, 0);
    return ;
  default:
    return ;
  }
}

/**
  Processor specific hook point at each SMM entry.

  @param  CpuIndex    The index of the cpu which need to check.
**/
VOID
SmmRendezvousExit (
  IN UINTN  CpuIndex
  )
{
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
    //
    // T-state fix
    // NHM & SNB BWG: T-state throttling should be disabled while executing the SMM handler.
    //
    AsmWriteMsr64 (EFI_MSR_IA32_CLOCK_MODULATION, gSmmCpuPrivate->TstateMsr[CpuIndex]);
    return ;
  default:
    return ;
  }
}

/**
  Initialize SMRR context in SMM Init.
**/
VOID
InitializeSmmMtrrManager (
  VOID
  )
{
  mThisCpu = GetCpuFamily ();
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
    mFeatureContext.SmrrEnabled = TRUE;
    return ;
  default:
    return ;
  }
}

/**
  Initialize SMRR register in SMM Relocate.

  @param  SmrrBase           The base address SMRR.
  @param  SmrrSize           The size of SMRR.
  @param  MsegBase           The base address MSEG.
  @param  IsBsp              If this processor treated as BSP.
**/
VOID
SmmInitiFeatures (
  IN UINT32  SmrrBase,
  IN UINT32  SmrrSize,
  IN UINT32  MsegBase,
  IN BOOLEAN IsBsp
  )
{
  mThisCpu = GetCpuFamily ();
  ASSERT (mThisCpu != NULL);

  switch (mThisCpu->Family) {
  case CpuSilvermont:
    if (!IsSmrrSupported ()) {
      return ;
    }
    NehalemInitSmrr (SmrrBase, SmrrSize);
    if (FeaturePcdGet (PcdCpuStmSupport)) {
      NehalemInitMseg (MsegBase, IsBsp);
    }
    return ;
  default:
    return ;
  }
}
