/*******************************************************************************
* Copyright 2015-2020 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
//   Description:
//       Intel(R) IPP RESIZE Transform Functions
//
//   Content:
//       ippiResizeGetSize_LT
//       ippiResizeNearestInit_LT
//       ippiResizeLinearInit_LT
//       ippiResizeCubicInit_LT
//       ippiResizeLanczosInit_LT
//       ippiResizeSuperInit_LT
//       ippiResizeGetBorderSize_LT
//       ippiResizeGetBufferSize_LT
//       ippiResizeNearest_8u_C1R_LT
//       ippiResizeNearest_8u_C3R_LT
//       ippiResizeNearest_8u_C4R_LT
//       ippiResizeNearest_16u_C1R_LT
//       ippiResizeNearest_16u_C3R_LT
//       ippiResizeNearest_16u_C4R_LT
//       ippiResizeNearest_16s_C1R_LT
//       ippiResizeNearest_16s_C3R_LT
//       ippiResizeNearest_16s_C4R_LT
//       ippiResizeNearest_32f_C1R_LT
//       ippiResizeNearest_32f_C3R_LT
//       ippiResizeNearest_32f_C4R_LT
//       ippiResizeLinear_8u_C1R_LT
//       ippiResizeLinear_8u_C3R_LT
//       ippiResizeLinear_8u_C4R_LT
//       ippiResizeLinear_16u_C1R_LT
//       ippiResizeLinear_16u_C3R_LT
//       ippiResizeLinear_16u_C4R_LT
//       ippiResizeLinear_16s_C1R_LT
//       ippiResizeLinear_16s_C3R_LT
//       ippiResizeLinear_16s_C4R_LT
//       ippiResizeLinear_32f_C1R_LT
//       ippiResizeLinear_32f_C3R_LT
//       ippiResizeLinear_32f_C4R_LT
//       ippiResizeLinear_64f_C1R_LT
//       ippiResizeLinear_64f_C3R_LT
//       ippiResizeLinear_64f_C4R_LT
//       ippiResizeCubic_8u_C1R_LT
//       ippiResizeCubic_8u_C3R_LT
//       ippiResizeCubic_8u_C4R_LT
//       ippiResizeCubic_16u_C1R_LT
//       ippiResizeCubic_16u_C3R_LT
//       ippiResizeCubic_16u_C4R_LT
//       ippiResizeCubic_16s_C1R_LT
//       ippiResizeCubic_16s_C3R_LT
//       ippiResizeCubic_16s_C4R_LT
//       ippiResizeCubic_32f_C1R_LT
//       ippiResizeCubic_32f_C3R_LT
//       ippiResizeCubic_32f_C4R_LT
//       ippiResizeLanczos_8u_C1R_LT
//       ippiResizeLanczos_8u_C3R_LT
//       ippiResizeLanczos_8u_C4R_LT
//       ippiResizeLanczos_16u_C1R_LT
//       ippiResizeLanczos_16u_C3R_LT
//       ippiResizeLanczos_16u_C4R_LT
//       ippiResizeLanczos_16s_C1R_LT
//       ippiResizeLanczos_16s_C3R_LT
//       ippiResizeLanczos_16s_C4R_LT
//       ippiResizeLanczos_32f_C1R_LT
//       ippiResizeLanczos_32f_C3R_LT
//       ippiResizeLanczos_32f_C4R_LT
//       ippiResizeSuper_8u_C1R_LT
//       ippiResizeSuper_8u_C3R_LT
//       ippiResizeSuper_8u_C4R_LT
//       ippiResizeSuper_16u_C1R_LT
//       ippiResizeSuper_16u_C3R_LT
//       ippiResizeSuper_16u_C4R_LT
//       ippiResizeSuper_16s_C1R_LT
//       ippiResizeSuper_16s_C3R_LT
//       ippiResizeSuper_16s_C4R_LT
//       ippiResizeSuper_32f_C1R_LT
//       ippiResizeSuper_32f_C3R_LT
//       ippiResizeSuper_32f_C4R_LT
//
*/

#include "piresizen_tl.h"


static const IppiSizeL RESIZE_NEAREST_MIN_TILE_SIZE = {1,1};
static const IppiSizeL RESIZE_LINEAR_MIN_TILE_SIZE  = {8,8};
static const IppiSizeL RESIZE_CUBIC_MIN_TILE_SIZE   = {16,16};
static const IppiSizeL RESIZE_LANCZOS_MIN_TILE_SIZE = {24,24};
static const IppiSizeL RESIZE_SUPER_MIN_TILE_SIZE   = {16,16};

int owniGetThreadNumber()
{
#ifdef USE_OMP
    return omp_get_thread_num();
#else
    return 0;
#endif
}

int owniResizeGetDataSize(IppDataType dataType)
{
    switch (dataType)
    {
    case ipp8u:
        return (int)sizeof(Ipp8u);
    case ipp16u:
    case ipp16s:
        return (int)sizeof(Ipp16u);
    case ipp32f:
        return (int)sizeof(Ipp32f);
    case ipp64f:
        return (int)sizeof(Ipp64f);
    default:
        return -1;
    }
}


void owniResizeGetTileByIndex(int index, IppiPointL splitImage, IppiSizeL tileSize, IppiSizeL tailSize, IppiPointL *pTileOffset, IppiSizeL *pTileSize)
{
    IppiPointL tileOffsetL = {0};
    owniGetTileParamsByIndex_LT(index, splitImage, tileSize, tailSize, &tileOffsetL, pTileSize);
    pTileOffset->x = (int)tileOffsetL.x;
    pTileOffset->y = (int)tileOffsetL.y;
}

IppStatus owniResizeGetTileBufferSize(const IppiResizeSpec_LT* pSpec, int numChannels, IppSizeL *pBufferSize)
{
    IppStatus status = ippStsNoErr;
    IppiSizeL t0 = {0}, t1 = {0}, t2 = {0}, t3 = {0};
    IppSizeL  s0 = 0,   s1 = 0,   s2 = 0,   s3 = 0;
    IppSizeL size = 0;
    IppiResizeSpec *pResizeSpec = (void*)((Ipp8u*)pSpec + sizeof(ResizeInfo));
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;

    /* compute work buffer for each thread */
    t0.width  = t2.width  = pResizeInfo->tileSize.width;
    t0.height = t1.height = pResizeInfo->tileSize.height;
    t1.width  = t3.width  = pResizeInfo->lastTile.width;
    t2.height = t3.height = pResizeInfo->lastTile.height;

    status = ippiResizeGetBufferSize_L(pResizeSpec, t1, numChannels, &s1);
    status = ippiResizeGetBufferSize_L(pResizeSpec, t2, numChannels, &s2);
    status = ippiResizeGetBufferSize_L(pResizeSpec, t3, numChannels, &s3);
    status = ippiResizeGetBufferSize_L(pResizeSpec, t0, numChannels, &s0);

    size = IPP_MAX(s0, IPP_MAX(s1, IPP_MAX(s2, s3)));
    size = IPP_ALIGNED_SIZE(size, 64);

    if (status >= 0)
    {
        *pBufferSize = size;
    }

    return status;
}


/* /////////////////////////////////////////////////////////////////////////////
//  Name:               ippiResizeGetSize_LT
//  Purpose:            Computes the size of Spec structure and temporal buffer for Resize transform
//
//  Parameters:
//    srcSize           Size of the input image (in pixels)
//    dstSize           Size of the output image (in pixels)
//    dataType          Data type {ipp8u|ipp16u|ipp16s|ipp32f} and ipp64f only for Linear interpolation
//    numChannels       Number of channels, possible values are 1 or 3 or 4
//    interpolation     Interpolation method
//    antialiasing      Supported values:
//                        0 - resizing without antialiasing
//                        1 - resizing with antialiasing
//    pSpecSize         Pointer to the size (in bytes) of the Spec structure
//    pInitBufSize      Pointer to the size (in bytes) of the temporal buffer
//
//  Return Values:
//    ippStsNoErr               Indicates no error
//    ippStsNullPtrErr          Indicates an error if one of the specified pointers is NULL
//    ippStsNoOperation         Indicates a warning if width or height of any image is zero
//    ippStsSizeErr             Indicates an error in the following cases:
//                              -  if width or height of the source or destination image is negative,
//                              -  if the source image size is less than a filter size of the chosen
//                                 interpolation method (except ippSuper),
//                              -  if one of the specified dimensions of the source image is less than
//                                 the corresponding dimension of the destination image (for ippSuper method only),
//    ippStsExceededSizeErr     Indicates an error in the following cases:
//                              -  if one of the calculated sizes exceeds maximum of IppSizeL type positive value
//                                 (the size of the one of the processed images is too large)
//                              -  if one of width or height of the destination image or the source image with borders
//                                 exceeds 536870911 (0x1FFFFFFF)
//    ippStsInterpolationErr    Indicates an error if interpolation has an illegal value
//    ippStsDataTypeErr         Indicates an error when dataType has an illegal value.
//    ippStsNoAntialiasing      Indicates a warning if specified interpolation does not support antialiasing
//    ippStsNotSupportedModeErr Indicates an error if requested mode is currently not supported
//
//  Notes:
//    1. Supported interpolation methods are ippNearest, ippLinear, ippCubic, ippLanczos and ippSuper.
//    2. If antialiasing value is equal to 1, use the ippResizeAntialiasing<Filter>Init functions, otherwise, use ippResize<Filter>Init
//    3. The implemented interpolation algorithms have the following filter sizes: Nearest Neighbor 1x1,
//       Linear 2x2, Cubic 4x4, 2-lobed Lanczos 4x4.
*/
IPPFUN (IppStatus, ippiResizeGetSize_LT, (
    IppiSizeL srcSize, IppiSizeL dstSize, IppDataType dataType,
    IppiInterpolationType interpolation, Ipp32u antialiasing,
    IppSizeL* pSpecSize, IppSizeL* pInitBufSize))
{
    IppStatus status = ippStsNoErr;

    status = ippiResizeGetSize_L (srcSize, dstSize, dataType, interpolation, antialiasing, pSpecSize, pInitBufSize);

    if (status >= 0)
    {
        *pSpecSize += sizeof(ResizeInfo);
        if (antialiasing != 0 && ((interpolation == ippNearest) || (interpolation == ippSuper))) return ippStsNoAntialiasing;
    }

    return status;
}


/* /////////////////////////////////////////////////////////////////////////////
//  Name:               ippiResizeNearestInit_LT
//                      ippiResizeLinearInit_LT
//                      ippiResizeCubicInit_LT
//                      ippiResizeLanczosInit_LT
//                      ippiResizeSuperInit_LT
//
//  Purpose:            Initializes the Spec structure for the Resize transform
//                      by different interpolation methods
//
//  Parameters:
//    srcSize           Size of the input image (in pixels)
//    dstSize           Size of the output image (in pixels)
//    dataType          Data type {ipp8u|ipp16u|ipp16s|ipp32f} and ipp64f only for Linear interpolation
//    numChannels       Number of channels, possible values are 1 or 3 or 4
//    valueB            The first parameter (B) for specifying Cubic filters
//    valueC            The second parameter (C) for specifying Cubic filters
//    numLobes          The parameter for specifying Lanczos (2 or 3) or Hahn (3 or 4) filters
//    pInitBuf          Pointer to the temporal buffer for several filter initialization
//    pSpec             Pointer to the Spec structure for resize filter
//
//  Return Values:
//    ippStsNoErr               Indicates no error
//    ippStsNullPtrErr          Indicates an error if one of the specified pointers is NULL
//    ippStsNoOperation         Indicates a warning if width or height of any image is zero
//    ippStsSizeErr             Indicates an error in the following cases:
//                              -  if width or height of the source or destination image is negative,
//                              -  if the source image size is less than a filter size of the chosen
//                                 interpolation method (except ippiResizeSuperInit).
//                              -  if one of the specified dimensions of the source image is less than
//                                 the corresponding dimension of the destination image
//                                 (for ippiResizeSuperInit only).
//    ippStsExceededSizeErr     Indicates an error if one of width or height of the destination image or
//                              the source image with borders exceeds 536870911 (0x1FFFFFFF)
//    ippStsDataTypeErr         Indicates an error when dataType has an illegal value.
//    ippStsNumChannelsErr      Indicates an error if numChannels has illegal value
//    ippStsNotSupportedModeErr Indicates an error if the requested mode is not supported.
//
//  Notes/References:
//    1. The equation shows the family of cubic filters:
//           ((12-9B-6C)*|x|^3 + (-18+12B+6C)*|x|^2                  + (6-2B)  ) / 6   for |x| < 1
//    K(x) = ((   -B-6C)*|x|^3 + (    6B+30C)*|x|^2 + (-12B-48C)*|x| + (8B+24C)) / 6   for 1 <= |x| < 2
//           0   elsewhere
//    Some values of (B,C) correspond to known cubic splines: Catmull-Rom (B=0,C=0.5), B-Spline (B=1,C=0) and other.
//      Mitchell, Don P.; Netravali, Arun N. (Aug. 1988). "Reconstruction filters in computer graphics"
//      http://www.mentallandscape.com/Papers_siggraph88.pdf
//
//    2. Hahn filter does not supported now.
//    3. The implemented interpolation algorithms have the following filter sizes: Nearest Neighbor 1x1,
//       Linear 2x2, Cubic 4x4, 2-lobed Lanczos 4x4, 3-lobed Lanczos 6x6.
*/
IPPFUN (IppStatus, ippiResizeNearestInit_LT, (IppiSizeL srcSize, IppiSizeL dstSize, IppDataType dataType, Ipp32u numChannels, IppiResizeSpec_LT* pSpec))
{
    IppStatus status = ippStsNoErr;
    ResizeInfo *pResizeInfo = 0;

    if (pSpec == 0) return ippStsNullPtrErr;
    if (numChannels != 1 && numChannels != 3 && numChannels != 4) return ippStsNumChannelsErr;

    pResizeInfo = (ResizeInfo*)pSpec;
    pSpec = (IppiResizeSpec_LT*)((Ipp8u*)pSpec + sizeof(ResizeInfo));

    status = ippiResizeNearestInit_L (srcSize, dstSize, dataType, (IppiResizeSpec*)pSpec);

    if (status >= 0 && status != ippStsNoOperation)
    {
        IppiSizeL tileSize;
        IppSizeL srcPixNumber = srcSize.width * srcSize.height;
        IppSizeL dstPixNumber = dstSize.width * dstSize.height;
        tileSize.width  = dstSize.width;
        tileSize.height = dstSize.height;
        if ( srcPixNumber >= RESIZE_TILE_THRESHOLD || dstPixNumber >= RESIZE_TILE_THRESHOLD )
        {
            owniGetTileSizeSimple_LT(dstSize, RESIZE_NEAREST_MIN_PIX_NUMBER / numChannels, RESIZE_NEAREST_MIN_TILE_SIZE, &tileSize);
        }

        ippiSplitToTiles_LT( dstSize, tileSize, &pResizeInfo->split, &pResizeInfo->tileSize, &pResizeInfo->lastTile );

        pResizeInfo->dataType    = dataType;
        pResizeInfo->srcSize     = srcSize;
        pResizeInfo->dstSize     = dstSize;
        pResizeInfo->numChannels = numChannels;
        pResizeInfo->interp      = ippNearest;

        status = ippiResizeGetBorderSize_L((IppiResizeSpec*)pSpec, &pResizeInfo->borderSize);
    }

    return status;
}

IPPFUN (IppStatus, ippiResizeLinearInit_LT, (IppiSizeL srcSize, IppiSizeL dstSize, IppDataType dataType, Ipp32u numChannels, IppiResizeSpec_LT* pSpec))
{
    IppStatus status = ippStsNoErr;
    ResizeInfo *pResizeInfo = 0;

    if (pSpec == 0) return ippStsNullPtrErr;
    if (numChannels != 1 && numChannels != 3 && numChannels != 4) return ippStsNumChannelsErr;

    pResizeInfo = (ResizeInfo*)pSpec;
    pSpec = (IppiResizeSpec_LT*)((Ipp8u*)pSpec + sizeof(ResizeInfo));

    status = ippiResizeLinearInit_L (srcSize, dstSize, dataType, (IppiResizeSpec*)pSpec);

    if (status >= 0 && status != ippStsNoOperation)
    {
        IppiSizeL tileSize;
        IppSizeL srcPixNumber = srcSize.width * srcSize.height;
        IppSizeL dstPixNumber = dstSize.width * dstSize.height;
        tileSize.width  = dstSize.width;
        tileSize.height = dstSize.height;
        if ( srcPixNumber >= RESIZE_TILE_THRESHOLD || dstPixNumber >= RESIZE_TILE_THRESHOLD )
        {
            owniGetTileSizeSimple_LT(dstSize, RESIZE_LINEAR_MIN_PIX_NUMBER / numChannels, RESIZE_LINEAR_MIN_TILE_SIZE, &tileSize);
        }

        ippiSplitToTiles_LT( dstSize, tileSize, &pResizeInfo->split, &pResizeInfo->tileSize, &pResizeInfo->lastTile );

        pResizeInfo->dataType    = dataType;
        pResizeInfo->srcSize     = srcSize;
        pResizeInfo->dstSize     = dstSize;
        pResizeInfo->numChannels = numChannels;
        pResizeInfo->interp      = ippLinear;

        status = ippiResizeGetBorderSize_L((IppiResizeSpec*)pSpec, &pResizeInfo->borderSize);
    }

    return status;
}

IPPFUN (IppStatus, ippiResizeCubicInit_LT, (IppiSizeL srcSize, IppiSizeL dstSize, IppDataType dataType, Ipp32u numChannels, Ipp32f valueB, Ipp32f valueC, IppiResizeSpec_LT* pSpec, Ipp8u* pInitBuf))
{
    IppStatus status = ippStsNoErr;
    ResizeInfo *pResizeInfo = 0;

    if (pSpec == 0) return ippStsNullPtrErr;
    if (numChannels != 1 && numChannels != 3 && numChannels != 4) return ippStsNumChannelsErr;

    pResizeInfo = (ResizeInfo*)pSpec;
    pSpec = (IppiResizeSpec_LT*)((Ipp8u*)pSpec + sizeof(ResizeInfo));

    status = ippiResizeCubicInit_L (srcSize, dstSize, dataType, valueB, valueC, (IppiResizeSpec*)pSpec, pInitBuf);

    if (status >= 0 && status != ippStsNoOperation)
    {
        IppiSizeL tileSize;
        IppSizeL srcPixNumber = srcSize.width * srcSize.height;
        IppSizeL dstPixNumber = dstSize.width * dstSize.height;
        tileSize.width  = dstSize.width;
        tileSize.height = dstSize.height;
        if ( srcPixNumber >= RESIZE_TILE_THRESHOLD || dstPixNumber >= RESIZE_TILE_THRESHOLD )
        {
            owniGetTileSizeSimple_LT(dstSize, RESIZE_CUBIC_MIN_PIX_NUMBER / numChannels, RESIZE_CUBIC_MIN_TILE_SIZE, &tileSize);
        }

        ippiSplitToTiles_LT( dstSize, tileSize, &pResizeInfo->split, &pResizeInfo->tileSize, &pResizeInfo->lastTile );

        pResizeInfo->dataType    = dataType;
        pResizeInfo->srcSize     = srcSize;
        pResizeInfo->dstSize     = dstSize;
        pResizeInfo->numChannels = numChannels;
        pResizeInfo->interp      = ippCubic;

        status = ippiResizeGetBorderSize_L((IppiResizeSpec*)pSpec, &pResizeInfo->borderSize);
    }

    return status;
}

IPPFUN (IppStatus, ippiResizeLanczosInit_LT, (IppiSizeL srcSize, IppiSizeL dstSize, IppDataType dataType, Ipp32u numChannels, Ipp32u numLobes, IppiResizeSpec_LT* pSpec, Ipp8u* pInitBuf))
{
    IppStatus status = ippStsNoErr;
    ResizeInfo *pResizeInfo = 0;

    if (pSpec == 0) return ippStsNullPtrErr;
    if (numChannels != 1 && numChannels != 3 && numChannels != 4) return ippStsNumChannelsErr;

    pResizeInfo = (ResizeInfo*)pSpec;
    pSpec = (IppiResizeSpec_LT*)((Ipp8u*)pSpec + sizeof(ResizeInfo));

    status = ippiResizeLanczosInit_L (srcSize, dstSize, dataType, numLobes, (IppiResizeSpec*)pSpec, pInitBuf);

    if (status >= 0 && status != ippStsNoOperation)
    {
        IppiSizeL tileSize;
        IppSizeL srcPixNumber = srcSize.width * srcSize.height;
        IppSizeL dstPixNumber = dstSize.width * dstSize.height;
        tileSize.width  = dstSize.width;
        tileSize.height = dstSize.height;
        if ( srcPixNumber >= RESIZE_TILE_THRESHOLD || dstPixNumber >= RESIZE_TILE_THRESHOLD )
        {
            owniGetTileSizeSimple_LT(dstSize, RESIZE_LANCZOS_MIN_PIX_NUMBER / numChannels, RESIZE_LANCZOS_MIN_TILE_SIZE, &tileSize);
        }

        ippiSplitToTiles_LT( dstSize, tileSize, &pResizeInfo->split, &pResizeInfo->tileSize, &pResizeInfo->lastTile );

        pResizeInfo->dataType    = dataType;
        pResizeInfo->srcSize     = srcSize;
        pResizeInfo->dstSize     = dstSize;
        pResizeInfo->numChannels = numChannels;
        pResizeInfo->interp      = ippLanczos;

        status = ippiResizeGetBorderSize_L((IppiResizeSpec*)pSpec, &pResizeInfo->borderSize);
    }

    return status;
}

IPPFUN (IppStatus, ippiResizeSuperInit_LT, (IppiSizeL srcSize, IppiSizeL dstSize, IppDataType dataType, Ipp32u numChannels, IppiResizeSpec_LT* pSpec))
{
    IppStatus status = ippStsNoErr;
    ResizeInfo *pResizeInfo = 0;

    if (pSpec == 0) return ippStsNullPtrErr;
    if (numChannels != 1 && numChannels != 3 && numChannels != 4) return ippStsNumChannelsErr;

    pResizeInfo = (ResizeInfo*)pSpec;
    pSpec = (IppiResizeSpec_LT*)((Ipp8u*)pSpec + sizeof(ResizeInfo));

    status = ippiResizeSuperInit_L (srcSize, dstSize, dataType, (IppiResizeSpec*)pSpec);

    if (status >= 0 && status != ippStsNoOperation)
    {
        IppiSizeL tileSize;
        IppSizeL srcPixNumber = srcSize.width * srcSize.height;
        IppSizeL dstPixNumber = dstSize.width * dstSize.height;
        tileSize.width  = dstSize.width;
        tileSize.height = dstSize.height;
        if ( srcPixNumber >= RESIZE_TILE_THRESHOLD || dstPixNumber >= RESIZE_TILE_THRESHOLD )
        {
            owniGetTileSizeSimple_LT(dstSize, RESIZE_SUPER_MIN_PIX_NUMBER / numChannels, RESIZE_SUPER_MIN_TILE_SIZE, &tileSize);
        }

        ippiSplitToTiles_LT( dstSize, tileSize, &pResizeInfo->split, &pResizeInfo->tileSize, &pResizeInfo->lastTile );

        pResizeInfo->dataType    = dataType;
        pResizeInfo->srcSize     = srcSize;
        pResizeInfo->dstSize     = dstSize;
        pResizeInfo->numChannels = numChannels;
        pResizeInfo->interp      = ippSuper;

        status = ippiResizeGetBorderSize_L((IppiResizeSpec*)pSpec, &pResizeInfo->borderSize);
    }

    return status;
}

/* /////////////////////////////////////////////////////////////////////////////
//  Name:               ippiResizeGetBorderSize_LT
//  Purpose:            Computes the size of possible borders for Resize transform
//
//  Parameters:
//    pSpec             Pointer to the Spec structure for resize filter
//    borderSize        Size of necessary borders (for memory allocation)
//
//  Return Values:
//    ippStsNoErr           Indicates no error
//    ippStsNullPtrErr      Indicates an error if one of the specified pointers is NULL
//    ippStsContextMatchErr Indicates an error if pointer to an invalid pSpec structure is passed
*/
IPPFUN (IppStatus, ippiResizeGetBorderSize_LT, (const IppiResizeSpec_LT* pSpec, IppiBorderSize* pBorderSize))
{
    return (pSpec != 0) ? ippiResizeGetBorderSize_L((IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)), pBorderSize) : ippStsNullPtrErr;
}

/* /////////////////////////////////////////////////////////////////////////////
//  Name:               ippiResizeGetBufferSize_LT
//  Purpose:            Computes the size of external buffer for Resize transform
//
//  Parameters:
//    pSpec             Pointer to the Spec structure for resize filter
//    pBufSize          Pointer to the size (in bytes) of the external buffer
//
//  Return Values:
//    ippStsNoErr           Indicates no error
//    ippStsNullPtrErr      Indicates an error if one of the specified pointers is NULL
//    ippStsContextMatchErr Indicates an error if pointer to an invalid pSpec structure is passed
//    ippStsExceededSizeErr   Indicates an error if one of the calculated sizes exceeds maximum of IppSizeL type
//                            positive value (the size of the one of the processed images is too large)
*/
IPPFUN (IppStatus, ippiResizeGetBufferSize_LT, (const IppiResizeSpec_LT* pSpec, IppSizeL* pBufSize))
{
    int numThreads = 0;
    Ipp32u numChannels = 0;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;

    if (pSpec == 0 || pBufSize == 0) return ippStsNullPtrErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    numChannels = pResizeInfo->numChannels;

    ippGetNumThreads_LT(&numThreads);

    /* compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (status >= 0)
    {
        *pBufSize = tileBufferSize * numThreads;
    }

    return status;

}

/* /////////////////////////////////////////////////////////////////////////////
//  Name:               ippiResizeNearest
//                      ippiResizeLinear
//                      ippiResizeCubic
//                      ippiResizeLanczos
//                      ippiResizeSuper
//
//  Purpose:            Changes an image size by different interpolation methods
//
//  Parameters:
//    pSrc              Pointer to the source image
//    srcStep           Distance (in bytes) between of consecutive lines in the source image
//    pDst              Pointer to the destination image
//    dstStep           Distance (in bytes) between of consecutive lines in the destination image
//    border            Type of the border
//    borderValue       Pointer to the constant value(s) if border type equals ippBorderConstant
//    pSpec             Pointer to the Spec structure for resize filter
//    pBuffer           Pointer to the work buffer
//
//  Return Values:
//    ippStsNoErr               Indicates no error
//    ippStsNullPtrErr          Indicates an error if one of the specified pointers is NULL
//    ippStsBorderErr           Indicates an error if border type has an illegal value
//    ippStsContextMatchErr     Indicates an error if pointer to an invalid pSpec structure is passed
//    ippStsNotSupportedModeErr Indicates an error if requested mode is currently not supported
//    ippStsStepErr             Indicates an error if the step value is not data type multiple
//
//  Notes:
//    1. Supported border types are ippBorderInMemory and ippBorderReplicate
//       (except Nearest Neighbor and Super Sampling methods).
//    2. Hahn filter does not supported now.
*/
/*****************************************************************************/
/* Nearest resize functions */
/*****************************************************************************/

IppStatus ippiResizeNearest_8u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp8u *pSrcRoi = 0; 
    Ipp8u       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_8u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

/* 8u Resize functions with Nearest interpolation */
IPPFUN (IppStatus, ippiResizeNearest_8u_C1R_LT, (const Ipp8u* pSrc, IppSizeL srcStep, Ipp8u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                    /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_8u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp8u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_8u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_8u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp8u *pSrcRoi = 0; 
    Ipp8u       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
    
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx); 
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
        threadStatus = ippiResizeNearest_8u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_8u_C3R_LT, (const Ipp8u* pSrc, IppSizeL srcStep, Ipp8u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */
    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;
    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;
    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_8u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }
    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp8u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_8u_C3R_LT_Fun);
    }
    return status;
}

IppStatus ippiResizeNearest_8u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp8u *pSrcRoi = 0; 
    Ipp8u       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx); 
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
        threadStatus = ippiResizeNearest_8u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_8u_C4R_LT, (const Ipp8u* pSrc, IppSizeL srcStep, Ipp8u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_8u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp8u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_8u_C4R_LT_Fun);
    }

    return status;
}


/* 16u Resize functions with Nearest interpolation */

IppStatus ippiResizeNearest_16u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp16u *pSrcRoi = 0; 
    Ipp16u       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 

    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx); 
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
        threadStatus = ippiResizeNearest_16u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_16u_C1R_LT, (const Ipp16u* pSrc, IppSizeL srcStep, Ipp16u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_16u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_16u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_16u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp16u *pSrcRoi = 0; 
    Ipp16u       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 

    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx); 
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
        threadStatus = ippiResizeNearest_16u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_16u_C3R_LT, (const Ipp16u* pSrc, IppSizeL srcStep, Ipp16u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_16u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_16u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_16u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp16u *pSrcRoi = 0; 
    Ipp16u       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_16u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_16u_C4R_LT, (const Ipp16u* pSrc, IppSizeL srcStep, Ipp16u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_16u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_16u_C4R_LT_Fun);
    }

    return status;
}


/* 16s Resize functions with Nearest interpolation */

IppStatus ippiResizeNearest_16s_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp16s *pSrcRoi = 0; 
    Ipp16s       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_16s_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_16s_C1R_LT, (const Ipp16s* pSrc, IppSizeL srcStep, Ipp16s* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_16s_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16s*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_16s_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_16s_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp16s *pSrcRoi = 0; 
    Ipp16s       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_16s_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_16s_C3R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_16s_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16s*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_16s_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_16s_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp16s *pSrcRoi = 0; 
    Ipp16s       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_16s_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_16s_C4R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_16s_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16s*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_16s_C4R_LT_Fun);
    }

    return status;
}


/* 32f Resize functions with Nearest interpolation */

IppStatus ippiResizeNearest_32f_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp32f *pSrcRoi = 0; 
    Ipp32f       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_32f_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_32f_C1R_LT, (const Ipp32f* pSrc, IppSizeL srcStep, Ipp32f* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_32f_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp32f*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_32f_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_32f_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp32f *pSrcRoi = 0; 
    Ipp32f       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_32f_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_32f_C3R_LT, (const Ipp32f* pSrc, IppSizeL srcStep, Ipp32f* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_32f_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp32f*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_32f_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeNearest_32f_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels; 
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
 
    const Ipp32f *pSrcRoi = 0; 
    Ipp32f       *pDstRoi = 0; 

    IppiPointL    roiOffset = { 0 }, srcRoiOffset = { 0 }; 
    IppiSizeL     roiSize = { 0 }; 
 
    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize); 
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset); 
 
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels); 
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x, roiOffset.y, numChannels); 
 
    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeNearest_32f_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeNearest_32f_C4R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeNearest_32f_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp32f*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeNearest_32f_C4R_LT_Fun);
    }

    return status;
}

/*****************************************************************************/
/* Linear resize functions */
/*****************************************************************************/

/* 8u Resize functions with Linear interpolation */

IppStatus ippiResizeLinear_8u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_8u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_8u_C1R_LT, (const Ipp8u* pSrc, IppSizeL srcStep, Ipp8u* pDst, IppSizeL dstStep, IppiBorderType border, const Ipp8u* pBorderValue, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_8u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u *)pSrc, (IppSizeL)srcStep, (Ipp8u *)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_8u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_8u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_8u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_8u_C3R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for  Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_8u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u *)pSrc, (IppSizeL)srcStep, (Ipp8u *)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_8u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_8u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_8u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_8u_C4R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_8u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u *)pSrc, (IppSizeL)srcStep, (Ipp8u *)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_8u_C4R_LT_Fun);
    }

    return status;
}


/* 16u Resize functions with Linear interpolation */

IppStatus ippiResizeLinear_16u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_16u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_16u_C1R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_16u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u *)pSrc, (IppSizeL)srcStep, (Ipp16u *)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_16u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_16u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_16u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_16u_C3R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_16u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u *)pSrc, (IppSizeL)srcStep, (Ipp16u *)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_16u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_16u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_16u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_16u_C4R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_16u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u *)pSrc, (IppSizeL)srcStep, (Ipp16u *)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_16u_C4R_LT_Fun);
    }

    return status;
}

/* 16s Resize functions with Linear interpolation */

IppStatus ippiResizeLinear_16s_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_16s_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_16s_C1R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_16s_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s *)pSrc, (IppSizeL)srcStep, (Ipp16s *)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_16s_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_16s_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_16s_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_16s_C3R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_16s_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s *)pSrc, (IppSizeL)srcStep, (Ipp16s *)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_16s_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_16s_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_16s_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_16s_C4R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_16s_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s *)pSrc, (IppSizeL)srcStep, (Ipp16s *)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_16s_C4R_LT_Fun);
    }

    return status;
}

/* 32f Resize functions with Linear interpolation */

IppStatus ippiResizeLinear_32f_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_32f_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_32f_C1R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;
    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_32f_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f *)pSrc, (IppSizeL)srcStep, (Ipp32f *)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_32f_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_32f_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_32f_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_32f_C3R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_32f_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f *)pSrc, (IppSizeL)srcStep, (Ipp32f *)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_32f_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_32f_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_32f_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_32f_C4R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_32f_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f *)pSrc, (IppSizeL)srcStep, (Ipp32f *)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_32f_C4R_LT_Fun);
    }

    return status;
}

/* 64f Resize functions with Linear interpolation */

IppStatus ippiResizeLinear_64f_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_64f_LT_Str * ts = (ippiResize_64f_LT_Str*)arg; 
    const Ipp64f* pSrc = (const Ipp64f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp64f* pDst = (Ipp64f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp64f* pBorderValue = (const Ipp64f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp64f *pSrcRoi = 0;
    Ipp64f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_64f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_64f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_64f_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_64f_C1R_LT, (
    const Ipp64f* pSrc, IppSizeL srcStep,
    Ipp64f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp64f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_64f_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_64f_LT_Str ts;
        resizeThreadingStructureEncode_64f((Ipp64f *)pSrc, (IppSizeL)srcStep, (Ipp64f *)pDst, (IppSizeL)dstStep, border, (Ipp64f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_64f_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_64f_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_64f_LT_Str * ts = (ippiResize_64f_LT_Str*)arg; 
    const Ipp64f* pSrc = (const Ipp64f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp64f* pDst = (Ipp64f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp64f* pBorderValue = (const Ipp64f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp64f *pSrcRoi = 0;
    Ipp64f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_64f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_64f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_64f_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_64f_C3R_LT, (
    const Ipp64f* pSrc, IppSizeL srcStep,
    Ipp64f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp64f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_64f_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_64f_LT_Str ts;
        resizeThreadingStructureEncode_64f((Ipp64f *)pSrc, (IppSizeL)srcStep, (Ipp64f *)pDst, (IppSizeL)dstStep, border, (Ipp64f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_64f_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLinear_64f_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_64f_LT_Str * ts = (ippiResize_64f_LT_Str*)arg; 
    const Ipp64f* pSrc = (const Ipp64f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp64f* pDst = (Ipp64f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp64f* pBorderValue = (const Ipp64f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
    /* Pointers to the source and destination tiles */
    const Ipp64f *pSrcRoi = 0;
    Ipp64f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_64f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_64f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = threadStatus = ippiResizeLinear_64f_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLinear_64f_C4R_LT, (
    const Ipp64f* pSrc, IppSizeL srcStep,
    Ipp64f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp64f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLinear_64f_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_64f_LT_Str ts;
        resizeThreadingStructureEncode_64f((Ipp64f *)pSrc, (IppSizeL)srcStep, (Ipp64f *)pDst, (IppSizeL)dstStep, border, (Ipp64f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLinear_64f_C4R_LT_Fun);
    }

    return status;
}

/*****************************************************************************/
/* Cubic resize functions */
/*****************************************************************************/

/* 8u Resize functions with Cubic interpolation */

IppStatus ippiResizeCubic_8u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_8u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_8u_C1R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_8u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_8u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_8u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_8u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_8u_C3R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_8u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_8u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_8u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_8u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_8u_C4R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_8u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_8u_C4R_LT_Fun);
    }

    return status;
}


/* 16u Resize functions with Cubic interpolation */

IppStatus ippiResizeCubic_16u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_16u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_16u_C1R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_16u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_16u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_16u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_16u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_16u_C3R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_16u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_16u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_16u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_16u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_16u_C4R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_16u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_16u_C4R_LT_Fun);
    }

    return status;
}

/* 16s Resize functions with Cubic interpolation */

IppStatus ippiResizeCubic_16s_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_16s_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_16s_C1R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_16s_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_16s_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_16s_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_16s_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_16s_C3R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_16s_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_16s_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_16s_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_16s_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_16s_C4R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_16s_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_16s_C4R_LT_Fun);
    }

    return status;
}

/* 32f Resize functions with Cubic interpolation */

IppStatus ippiResizeCubic_32f_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_32f_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_32f_C1R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;
    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_32f_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_32f_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_32f_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_32f_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_32f_C3R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_32f_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_32f_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeCubic_32f_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;

    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeCubic_32f_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeCubic_32f_C4R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeCubic_32f_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeCubic_32f_C4R_LT_Fun);
    }

    return status;
}

/*****************************************************************************/
/* Lanczos resize functions */
/*****************************************************************************/
/* 8u Resize functions with Lanczos interpolation */

IppStatus ippiResizeLanczos_8u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_8u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_8u_C1R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_8u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_8u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_8u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_8u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_8u_C3R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_8u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_8u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_8u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp8u* pBorderValue = (const Ipp8u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_8u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_8u_C4R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp8u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_8u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                       pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, border, (Ipp8u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_8u_C4R_LT_Fun);
    }

    return status;
}


/* 16u Resize functions with Lanczos interpolation */

IppStatus ippiResizeLanczos_16u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_16u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_16u_C1R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_16u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_16u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_16u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_16u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_16u_C3R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_16u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_16u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_16u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16u* pBorderValue = (const Ipp16u *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_16u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_16u_C4R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16u* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_16u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, border, (Ipp16u *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_16u_C4R_LT_Fun);
    }

    return status;
}

/* 16s Resize functions with Lanczos interpolation */

IppStatus ippiResizeLanczos_16s_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_16s_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_16s_C1R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_16s_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_16s_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_16s_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_16s_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_16s_C3R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_16s_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_16s_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_16s_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp16s* pBorderValue = (const Ipp16s *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_16s_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_16s_C4R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp16s* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_16s_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, border, (Ipp16s *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_16s_C4R_LT_Fun);
    }

    return status;
}

/* 32f Resize functions with Lanczos interpolation */

IppStatus ippiResizeLanczos_32f_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_32f_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_32f_C1R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;
    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;
 
    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_32f_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_32f_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_32f_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_32f_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_32f_C3R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_32f_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_32f_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeLanczos_32f_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppiBorderType border = ts->border;
    const Ipp32f* pBorderValue = (const Ipp32f *)ts->pBorderValue;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 
     /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL    roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL     roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeLanczos_32f_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, border,
                    pBorderValue, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeLanczos_32f_C4R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    IppiBorderType border, const Ipp32f* pBorderValue,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeLanczos_32f_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, border,
                                        pBorderValue, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, border, (Ipp32f *)pBorderValue, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeLanczos_32f_C4R_LT_Fun);
    }

    return status;
}

/*****************************************************************************/
/* Super resize functions */
/*****************************************************************************/

/* 8u Resize functions with Super interpolation */

IppStatus ippiResizeSuper_8u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_8u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_8u_C1R_LT, (const Ipp8u* pSrc, IppSizeL srcStep, Ipp8u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_8u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }


    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp8u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_8u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_8u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_8u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_8u_C3R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_8u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp8u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_8u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_8u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_8u_LT_Str * ts = (ippiResize_8u_LT_Str*)arg; 
    const Ipp8u* pSrc = (const Ipp8u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp8u* pDst = (Ipp8u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp8u *pSrcRoi = 0;
    Ipp8u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_8u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_8u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_8u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_8u_C4R_LT, (
    const Ipp8u* pSrc, IppSizeL srcStep,
    Ipp8u* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_8u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_8u_LT_Str ts;
        resizeThreadingStructureEncode_8u((Ipp8u*)pSrc, (IppSizeL)srcStep, (Ipp8u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp8u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_8u_C4R_LT_Fun);
    }

    return status;
}


/* 16u Resize functions with Super interpolation */

IppStatus ippiResizeSuper_16u_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_16u_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_16u_C1R_LT, (const Ipp16u* pSrc, IppSizeL srcStep, Ipp16u* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_16u_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_16u_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_16u_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_16u_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_16u_C3R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_16u_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_16u_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_16u_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16u_LT_Str * ts = (ippiResize_16u_LT_Str*)arg; 
    const Ipp16u* pSrc = (const Ipp16u*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16u* pDst = (Ipp16u*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16u *pSrcRoi = 0;
    Ipp16u       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16u(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16u(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_16u_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_16u_C4R_LT, (
    const Ipp16u* pSrc, IppSizeL srcStep,
    Ipp16u* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_16u_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16u_LT_Str ts;
        resizeThreadingStructureEncode_16u((Ipp16u*)pSrc, (IppSizeL)srcStep, (Ipp16u*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16u*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_16u_C4R_LT_Fun);
    }

    return status;
}


/* 16s Resize functions with Super interpolation */

IppStatus ippiResizeSuper_16s_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_16s_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_16s_C1R_LT, (const Ipp16s* pSrc, IppSizeL srcStep, Ipp16s* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_16s_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16s*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_16s_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_16s_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_16s_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_16s_C3R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_16s_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16s*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_16s_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_16s_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_16s_LT_Str * ts = (ippiResize_16s_LT_Str*)arg; 
    const Ipp16s* pSrc = (const Ipp16s*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp16s* pDst = (Ipp16s*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp16s *pSrcRoi = 0;
    Ipp16s       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_16s(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_16s(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_16s_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_16s_C4R_LT, (
    const Ipp16s* pSrc, IppSizeL srcStep,
    Ipp16s* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_16s_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_16s_LT_Str ts;
        resizeThreadingStructureEncode_16s((Ipp16s*)pSrc, (IppSizeL)srcStep, (Ipp16s*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp16s*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_16s_C4R_LT_Fun);
    }

    return status;
}

/* 32f Resize functions with Super interpolation */

IppStatus ippiResizeSuper_32f_C1R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_32f_C1R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_32f_C1R_LT, (const Ipp32f* pSrc, IppSizeL srcStep, Ipp32f* pDst, IppSizeL dstStep, const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 1;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_32f_C1R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp32f*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_32f_C1R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_32f_C3R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_32f_C3R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_32f_C3R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 3;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_32f_C3R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp32f*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_32f_C3R_LT_Fun);
    }

    return status;
}

IppStatus ippiResizeSuper_32f_C4R_LT_Fun(IppSizeL i, void* arg)
{ 
    ippiResize_32f_LT_Str * ts = (ippiResize_32f_LT_Str*)arg; 
    const Ipp32f* pSrc = (const Ipp32f*)ts->pSrc; 
    IppSizeL srcStep = ts->srcStep; 
    Ipp32f* pDst = (Ipp32f*)ts->pDst;
    IppSizeL dstStep = ts->dstStep; 
    Ipp8u* pBuffer = ts->pBuffer; 
    Ipp32u numChannels = ts->numChannels;
    IppSizeL tileBufferSize = ts->tileBufferSize; 
    IppiResizeSpec *pResizeSpec = ts->pResizeSpec; 
    ResizeInfo *pResizeInfo = ts->pResizeInfo;
    int threadIdx = 0;
 
    IppStatus threadStatus; 

    /* Pointers to the source and destination tiles */
    const Ipp32f *pSrcRoi = 0;
    Ipp32f       *pDstRoi = 0;
    /* Source and destination tile ROI parameters */
    IppiPointL     roiOffset  = {0}, srcRoiOffset = {0};
    IppiSizeL      roiSize    = {0};

    owniResizeGetTileByIndex((int)i, pResizeInfo->split, pResizeInfo->tileSize, pResizeInfo->lastTile, &roiOffset, &roiSize);
    threadStatus = ippiResizeGetSrcOffset_L(pResizeSpec, roiOffset, &srcRoiOffset);

    /* Compute pointers to ROIs */
    pSrcRoi = owniGetImagePointer_32f(pSrc, srcStep, srcRoiOffset.x, srcRoiOffset.y, numChannels);
    pDstRoi = owniGetImagePointer_32f(pDst, dstStep, roiOffset.x,    roiOffset.y,    numChannels);

    if (threadStatus >= 0) 
    { 
        ippGetThreadIdx_LT(&threadIdx);  
        Ipp8u *pTileBuffer = pBuffer + tileBufferSize * threadIdx; 
 
        threadStatus = ippiResizeSuper_32f_C4R_L(pSrcRoi, (IppSizeL)srcStep, pDstRoi, (IppSizeL)dstStep, roiOffset, roiSize, pResizeSpec, pTileBuffer);
    } 
    return threadStatus; 
}

IPPFUN (IppStatus, ippiResizeSuper_32f_C4R_LT, (
    const Ipp32f* pSrc, IppSizeL srcStep,
    Ipp32f* pDst, IppSizeL dstStep,
    const IppiResizeSpec_LT* pSpec, Ipp8u* pBuffer))
{
    Ipp32u numChannels = 4;
    IppStatus status = ippStsNoErr;
    IppSizeL numTiles = 0;
    IppSizeL tileBufferSize = 0;
    IppiResizeSpec *pResizeSpec = (IppiResizeSpec*)((Ipp8u*)pSpec + sizeof(ResizeInfo)); /* Resize Spec structure for Intel(R) IPP function call */
    ResizeInfo *pResizeInfo = (ResizeInfo*)pSpec;                 /* Resize Info structure */

    /* Bad argument checking */
    if (pSrc == 0 || pDst == 0 || pSpec == 0 || pBuffer == 0) return ippStsNullPtrErr;
    if (pResizeInfo->numChannels < numChannels)              return ippStsContextMatchErr;

    numTiles = pResizeInfo->split.x * pResizeInfo->split.y;

    /* Compute work buffer for one thread */
    status = owniResizeGetTileBufferSize(pSpec, numChannels, &tileBufferSize);
    if (status < 0) return status;

    if (numTiles == 1)
    {
        /* Source and destination ROI parameters */
        IppiPointL     roiOffset  = {0};
        /* Intel(R) IPP function call */
        return ippiResizeSuper_32f_C4R_L(pSrc, (IppSizeL)srcStep, pDst, (IppSizeL)dstStep, roiOffset, pResizeInfo->tileSize, pResizeSpec, pBuffer);
    }

    /* Image can be processed by Intel(R) IPP functions directly */
    {
        ippiResize_32f_LT_Str ts;
        resizeThreadingStructureEncode_32f((Ipp32f*)pSrc, (IppSizeL)srcStep, (Ipp32f*)pDst, (IppSizeL)dstStep, (IppiBorderType)0, (Ipp32f*)0, pBuffer, numChannels, tileBufferSize, pResizeSpec, pResizeInfo, &ts);
        status = ippParallelFor_LT(numTiles, (void*)&ts, ippiResizeSuper_32f_C4R_LT_Fun);
    }

    return status;
}
